<?php
namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Ticket;
use App\Models\User;
use Illuminate\Support\Facades\Mail;

class TicketController extends Controller
{
    public function index()
    {
        $tickets = Ticket::all();
        return view('tickets.index', compact('tickets'));
    }

    public function create()
    {
        return view('tickets.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'user_id' => 'required|exists:users,id',
            'status' => 'required|string',
        ]);

        // Auto-assign based on issue type (simplified example)
        $serviceOwner = User::where('role', 'service_owner')->inRandomOrder()->first();

        $ticket = Ticket::create([
            'title' => $request->title,
            'description' => $request->description,
            'user_id' => $request->user_id,
            'service_owner_id' => $serviceOwner->id,
            'status' => $request->status,
        ]);

        // Notify user (simplified example)
        // Mail::raw('Your ticket has been created.', function ($message) use ($ticket) {
        //     $message->to($ticket->user->email)->subject('Ticket Created');
        // });

        return redirect()->route('tickets.index')->with('success', 'Ticket created successfully.');
    }

    public function edit($id)
    {
        $ticket = Ticket::findOrFail($id);
        return view('tickets.edit', compact('ticket'));
    }

    public function update(Request $request, Ticket $ticket)
    {
        $data = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'status' => 'required|string',
            'service_owner_id' => 'required|exists:users,id'
        ]);

        $ticket->update($data);

        // Send update to the user when the ticket is assigned
        if ($ticket->wasChanged('service_owner_id')) {
            $this->sendWhatsAppNotification(
                $ticket->user->phone,
                $ticket->id,
                $ticket->updated_at->format('Y-m-d H:i:s'),
                '83972'
            );

            // Notify the service owner
            $this->notifyServiceOwner($ticket);
        }

        return redirect()->route('tickets.index')->with('success', 'Ticket updated successfully.');
    }

    private function sendWhatsAppNotification($phoneNumber, $ticketNumber, $updatedDate, $templateId)
    {
        $apiToken = '4969|RhZtToATIqzYkPwVqmlSqL7k6QuaUxmbnF3axFAE';

        Http::post('https://dash.botbiz.io/api/v1/whatsapp/send/template', [
            'apiToken' => $apiToken,
            'phoneNumberID' => '354530977737384',
            'botTemplateID' => $templateId,
            'templateVariable-ticketno-2' => $ticketNumber,
            'templateVariable-system-delivery-date-3' => $updatedDate
        ]);
    }

    private function notifyServiceOwner(Ticket $ticket)
    {
        $apiToken = '4969|RhZtToATIqzYkPwVqmlSqL7k6QuaUxmbnF3axFAE';
        $serviceOwner = $ticket->service_owner;

        Http::post('https://dash.botbiz.io/api/v1/whatsapp/send/template', [
            'apiToken' => $apiToken,
            'phoneNumberID' => '354530977737384',
            'botTemplateID' => '83976',
            'templateVariable-assignedTo-1' => $serviceOwner->name,
            'templateVariable-ticketOwner-2' => $ticket->user->name,
            'templateVariable-ticketDetails-3' => $ticket->description,
            'templateVariable-ticketno-4' => $ticket->id,
            'templateVariable-ticketOwner-5' => $ticket->user->name,
            'templateVariable-ticketOwnerPhone-6' => $ticket->user->phone,
            'sendToPhoneNumber' => $serviceOwner->phone
        ]);
    }

    public function destroy($id)
    {
        $ticket = Ticket::findOrFail($id);
        $ticket->delete();

        return redirect()->route('tickets.index')->with('success', 'Ticket deleted successfully.');
    }
}
